/*************************************************************************
 * The contents of this file are subject to the MYRICOM MX AND GM-2      *
 * MAPPING SOFTWARE AND DOCUMENTATION LICENSE (the "License"); User may  *
 * not use this file except in compliance with the License.  The full    *
 * text of the License can found in mapper directory in LICENSE.TXT      *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#include "lx.h"
#include "lx_net.h"
#include "lx_routing.h"

int d_rf;
int d_bn, d_rt, d_et;
int r1, r2, race;

static void
lx_map_update_state (lx_t*lx, int iport, lx_map_t*m)
{
  insist (lx);
  insist (m);
  
  mi_set_map_version (lx, iport, m->map_address, m->map_version,
                      m->num_hosts, m->done_everywhere, m->map_valid,
		      m->kids_match);

  /* Also do update for the other port, if there is one */
  if (lx->num_ports > 1)
  {
    m = lx_get_map (lx, iport ? 0 : 1);
    if (m != 0)
    {
      mi_set_map_version (lx, iport ? 0 : 1, m->map_address, m->map_version,
			  m->num_hosts, m->done_everywhere, m->map_valid,
			  m->kids_match);
    }
  }
  except:;
}

static int
lx_map_route_nodes (lx_t*lx, lx_map_t*m)
{
  int i;

  insist (lx);
  insist (m);
  for (i = 0; i < lx->num_ports; i++)
  {
    if (lx_get_node (m->root, i) &&
	!lx_routing_shortest_path (m, i, m->root, 0))
      return 0;
  }
  
  return 1;
  except: return 0;
}

static int
lx_map_shared (lx_t*lx)
{
  if (MI_MY_SIZE == 1) return 1;
  
  return lx_get_map (lx, 0) == lx_get_map (lx, 1);
}

static lx_map_t *
lx_map_get_other (lx_t*lx, lx_map_t*m)
{
  insist (lx);
  insist (m);
  
  if (MI_MY_SIZE == 1) return m;
  return &lx->maps [m == &lx->maps [0] ? 1 : 0];

  except: return 0;
}

static void
lx_map_check_other (lx_t*lx, lx_map_t*m)
{
  insist (lx);
  insist (m);

  if (MI_MY_SIZE == 1) return;
  
  if (lx_get_node (m->root, 0) && lx_get_node (m->root, 1))
  {
    mi_c (("map %d has both ports. marking other map invalid",
	   m == &lx->maps [0] ? 0 : 1));
    m = lx_map_get_other (lx, m);
    m->map_valid = 0;
    lx_bzero ((char*) m->map_address, 6);
    m->map_version = 0;
  }
  except:;
}


static int
lx_map_check_done_everywhere (lx_t*lx)
{
  int i;
  lx_map_t*m;
  
  insist (lx);

  for (i = 0; i < lx->num_ports; i++)
  {
    m = lx_get_map (lx, i);
    insist (m);

    if (!m->map_valid || !m->done_everywhere)
      return 0;
  }
  return 1;
  except: return 0;
}

void
lx_map_receive_callback (lx_t*lx, int iport, lx_packet_t*p, int length)
{
  int higher_than_me = 0;
  int r;
  lx_map_t*map = lx_get_map (lx, iport);
  
  insist (lx);
  insist (p);
  insist (iport >= 0);
  insist (iport < MI_MY_SIZE);
  insist (map);
  
  if (length < 4 || p->scout.header.type != mi_htons (LX_TYPE))
    return;
    
  higher_than_me = lx_higher_than_me (lx, p->scout.header.sender_address,
                                      p->scout.header.level);
  if (!higher_than_me && (p->change_report.header.subtype == mi_htons (LX_CHANGE_REPORT) || p->change_report.header.subtype == mi_htons (LX_BIG_CHANGE_REPORT)))
  {
    if (lx_memcmp (p->change_report.header.sender_address,
                   lx->changes [iport].child_address, 6) ||
	mi_htonl (p->change_report.header.timestamp) !=
	lx->changes [iport].timestamp)
    {
      lx_set_change (lx,
                     iport,
		     p->change_report.header.sender_address,
                     mi_htonl (p->change_report.header.timestamp),
		     p->change_report.change,
		     mi_htonl (p->change_report.map_version),
		     p->change_report.map_address,
		     p->change_report.header.echo_address,
		     p->change_report.mac_address,
		     mi_htons (p->change_report.node),
		     p->change_report.port,
		     p->change_report.header.subtype == mi_htons (LX_CHANGE_REPORT) ? 0 : mi_htonl (p->big_change_report.xid));
    }

    if (lx->pendings [LX_REPLY_SEND] == LX_NUM_REPLIES)
    {
      mi_c (("dropped reply"));
    }
    else
    {
      r =  lx_send_change_report_reply (lx, iport, &p->change_report);
      insist (r != 0);
    }

    return;
  }
  if (p->scout.header.subtype == mi_htons (LX_SCOUT))
  { 
    insist ((unsigned) length >= sizeof (lx_scout_message_t));

    mi_c (("scouted by " lx_mac_format " (level %d, reason %d, iport %d)",
	   lx_mac_args (p->scout.header.sender_address),
	   p->scout.header.level, p->scout.header.reason, iport));

    
    if (p->scout.header.reason == LX_SCOUTING)
    {
      if (map->mapping && higher_than_me)
      {
	mi_memcpy (map->winner_address, p->scout.header.sender_address, 6);
	map->winner_route.length = p->scout.route_length;
	map->winner_iport = iport;
	lx_copy_route (map->winner_route.hops, p->scout.route,
	               map->winner_route.length);
	
	mi_c (("my winner is " lx_mac_format " (level %d)",
	       lx_mac_args (p->scout.header.sender_address),
	       p->scout.header.level));
	map->map_valid = 0;
	lx_map_update_state (lx, iport, map);
	lx->aborted = 1;
      }
      else if (map->map_valid &&
               !lx_bsearch_host (map, p->scout.header.sender_address,
	                         p->scout.header.level))
      {
#ifndef MI_XM
	lx_node_t*dc;
	insist (!(dc = lx_find_host (lx, &map->hosts,
	                             p->scout.header.sender_address,
				     p->scout.header.level, &dc)) ||
		lx_host_c (dc)->level != p->scout.header.level);
#endif
	mi_c (("he is not in my map. marking my map as invalid"));
	lx_set_change (lx, iport, 0, 0, LX_SCOUT_NOT_IN_MAP,
	               map->map_version, map->map_address,
		       lx->mac_address, p->scout.header.sender_address, 0, 0, 0);
	r1 = 1;
	map->map_valid = 0;
	lx_map_update_state (lx, iport, map);
      }
    }
    else if ((p->scout.header.reason == LX_TREE) && !map->active)
    {
      int done_everywhere;
      
      mi_memcpy (map->parent_address, p->scout.header.sender_address, 6);
      map->parent_route.length = p->scout.route_length;
      map->parent_iport = iport;
      lx_copy_route (map->parent_route.hops, p->scout.route,
                     map->parent_route.length);

      mi_c (("my parent is " lx_mac_format " (level %d)",
	     lx_mac_args (p->scout.header.sender_address),
	     p->scout.header.level));
      if (lx_hostcmp (p->scout.header.sender_address,
                      p->scout.header.level, lx->mac_address, lx->level) <= 0)
      {
	mi_c (("my parent is evil. this map can't be valid"));
	map->map_valid = 0;
	lx_map_update_state (lx, iport, map);
      }
      done_everywhere = (map->map_valid &&
                         (mi_htons (p->scout.flags) & LX_FLAG_DONE_EVERYWHERE))
	!= 0;
      
      if (done_everywhere != map->done_everywhere)
      {
	map->done_everywhere = done_everywhere;
	lx_map_update_state (lx, iport, map);
      }      
    }
  }
  if (p->scout.header.subtype == mi_htons (LX_SCOUT))
  {
    if (higher_than_me)
    {
      int pause = (mi_htons (p->scout.flags) & LX_FLAG_PAUSE) != 0;
      if (pause ||
          lx_zero (lx->pause_address, 6) ||
	  !lx_memcmp (p->scout.header.sender_address, lx->pause_address, 6))
      {
	if ((lx->net_paused = pause))
	{
	  mi_c ((lx_mac_format " is pausing me",
		 lx_mac_args (p->scout.header.sender_address)));
	  mi_memcpy (lx->pause_address, p->scout.header.sender_address, 6);
	}
      }

      if (map->map_valid && !lx_memcmp (map->map_address, lx->mac_address, 6))
      {
	mi_memcpy (map->winner_address, p->scout.header.sender_address, 6);
	map->winner_route.length = p->scout.route_length;
	map->winner_iport = iport;
	lx_copy_route (map->winner_route.hops, p->scout.route, map->winner_route.length);
	
	mi_c (("my winner is " lx_mac_format " (level %d)", lx_mac_args (p->scout.header.sender_address),
	       p->scout.header.level));

	map->map_valid = 0;
	map->active = 0;
	lx_map_update_state (lx, iport, map);
	lx->aborted = 1;
      }
    }
    else
    {
      if (map->map_version == mi_htonl (p->scout.map_version) &&
          !lx_memcmp (p->scout.map_address, map->map_address, 6) &&
	  map->map_valid &&
	  !(mi_htons (p->scout.flags) & LX_FLAG_MAP_VALID))
      {
	mi_c (("my child is telling me my map is invalid."));
	map->map_valid = 0;
	lx_map_update_state (lx, iport, map);
      }
    }
    
    if (lx->pendings [LX_REPLY_SEND] == LX_NUM_REPLIES)
    {
      mi_c (("dropped reply"));
      r = 1;
    }
    else if (mi_htons(p->scout.flags) & LX_FLAG_MCP_REPLIED)
    {
      mi_c (("MCP replied for us"));
      r = 1;
    }
    else
    {
      r = lx_send_scout_reply (lx, iport, &p->scout);
      insist (r != 0);
    }
  }
  else if (p->scout_reply.header.subtype == mi_htons (LX_SCOUT_REPLY))
  {
#if 0
    mi_c (("got reply message %d from " lx_mac_format,
	   mi_htonl (p->scout.header.timestamp),
	   lx_mac_args (p->scout.header.sender_address)));
#endif
  }

  else if (p->map_request.header.subtype == mi_htons (LX_MAP_REQUEST))
  {
    mi_c (("got map request %d from " lx_mac_format,
	   mi_htonl (p->scout.header.timestamp),
	   lx_mac_args (p->map_request.header.sender_address)));

    if (lx->aborted || !map->map_valid)
    {
      mi_c (("I don't have a map to give."));
    }
    else
    {
      if (lx->pendings [LX_BIG_SEND] == LX_NUM_BIGS)
      {
	mi_c (("no map buffers, he'll have to ask again"));
      }
      else
      {
	r = lx_send_map_reply (lx, iport, map, &p->map_request);
	insist (r != 0);
      }
    }
  }
  except:;
}

static lx_node_t *
lx_map_add_host (lx_map_t*m,
                 int iport,
		 lx_node_t*n,
		 signed char port,
		 int oiport,
		 int host_type,
		 int level,
		 unsigned char mac_address [6],
		 lx_node_t*higher)
{
  lx_node_t*nn;
  lx_route_t route;
  lx_route_t *rp;
  
  insist (m && n && mac_address);
  insist (!lx_zero (mac_address, 6));
  insist (iport >= 0 && iport < MI_MY_SIZE);
  insist (oiport >= 0 && oiport < LX_HOST_SIZE);

  mi_c (("found host " lx_mac_format " (type %d) on port %d",
         lx_mac_args (mac_address), host_type, port));

  insist (port > -LX_XBAR_SIZE);
  insist (port < LX_XBAR_SIZE);
  
  if ((nn = lx_get_node (n, port)))
  {
    insist (!nn->xbar);
    insist (lx_memcmp (lx_host_c (nn)->mac_address, mac_address, 6) == 0);
    mi_c (("already found"));
    return 0;
  }

  rp = lx_append_route (&route, &lx_xbar_c (n)->routes [iport], port);
  nn = lx_new_host (m, iport, oiport, mac_address, host_type, level, rp);
  insist (nn);
  lx_insert (&m->hosts, higher, nn);
  lx_connect (n, port, nn, oiport);
  insist (m->hosts.head == m->root);
  
  return nn;

  except: return 0;
}

static lx_node_t*lx_map_add_xbar (lx_map_t*m, int iport, lx_node_t*n, signed char port, unsigned id, int mask, int absolute_port)
{
  lx_node_t*nn;
  lx_route_t route;
  lx_route_t *rp;

  insist (iport >= 0 && iport < MI_MY_SIZE);
  insist (m && n);
  
  mi_c (("found xbar on port %d", port));

  insist (port > -LX_XBAR_SIZE && port < LX_XBAR_SIZE);

  if ((nn = lx_get_node (n, port)))
  {
    insist (nn->xbar);
    mi_c (("already found"));
    return nn;
  }

  rp = lx_append_route (&route, &lx_xbar_c (n)->routes [iport], port);
  nn = lx_new_xbar (m, iport, rp, id, mask, absolute_port);
  insist (nn);
  return lx_connect (n, port, nn, 0) ? nn : 0;

  except: return 0;
}

static int
lx_map_merge (lx_map_t*m, lx_node_t*copy, lx_node_t*original, int offset)
{
  lx_node_t*n;
  int i, p;

  insist (m && copy && original);
  insist (!m->merge_queue.head);
  insist (!copy->original.index);
  insist (copy != original);
  insist (copy->xbar && original->xbar);

  mi_c (("merging duplicate " lx_node_format " with original " lx_node_format,
	 lx_node_args (copy), lx_node_args (original)));
  
  insist (mi_match (lx_iport (m),
                    &lx_xbar_c (copy)->routes [lx_iport (m)],
                    &lx_xbar_c (original)->routes [lx_iport (m)]));

  lx_remove_from_anywhere (copy);
  lx_put (&m->merge_queue, copy);
  copy->original.index = mi_node2index (original);
  copy->original.port = offset;
  insist (mi_index2node (copy->original.index) == original);
  
  while ((n = (lx_node_t*) lx_get (&m->merge_queue)))
  {
    lx_node_t*nn, *oo;
    lx_node_t*no = mi_index2node (n->original.index);
    insist (no);
    
    mi_punt (m->lx);

    for (i = 1 - LX_XBAR_SIZE; i < LX_XBAR_SIZE; i++)
    {
      mi_punt (m->lx);

      p = i + n->original.port;
      if ((nn = lx_get_node (n, i)))
      {
	mi_punt (m->lx);

	if (!(oo = lx_get_node (no, p)))
	{
	  mi_punt (m->lx);

	  if (!lx_connect (no, p, nn, lx_get_port (n, i)))
	    return 0;
	}
	else
	{
	  mi_punt (m->lx);

	  if (nn != oo && !oo->original.index && nn->xbar == oo->xbar)
	  {
	    mi_c ((lx_node_format " matches " lx_node_format, 
		   lx_node_args (nn),
		   lx_node_args (oo)));
	    
	    mi_punt (m->lx);

	    if (!nn->xbar &&
	        lx_memcmp (lx_host_c (nn)->mac_address,
		           lx_host_c (oo)->mac_address, 6))
	      return -1;

	    insist (mi_match (lx_iport (m),
	                      &lx_xbar_c (nn)->routes [lx_iport (m)],
			      &lx_xbar_c (oo)->routes [lx_iport (m)]));
	    mi_punt (m->lx);

	    lx_remove_from_anywhere (nn);

	    if (nn->xbar)
	    {
	      nn->original.index = mi_node2index (oo);
	      insist (mi_index2node (nn->original.index));
	      nn->original.port = lx_get_port (no, p)  - lx_get_port (n, i);
	      insist (nn->original.port > - 2 * LX_XBAR_SIZE);
	      insist (nn->original.port < 2 * LX_XBAR_SIZE);
	      lx_put (&m->merge_queue, nn);
	    }
	    else
	      lx_delete_node (m, nn);
	  }
	}
      }
    }
    lx_delete_node (m, n);
  }

  return 1;
  except: return 0;
}

static int
lx_map_found_host (lx_map_t*m,
                   lx_node_t*n,
		   int iport,
		   int oiport,
		   int port,
		   int host_type,
		   unsigned char mac_address [6],
		   unsigned map_version,
		   unsigned char map_address [6],
		   int level,
		   int flags,
		   int context)
{
  lx_node_t*h, *higher;
  lx_route_t r;
  lx_t*lx = m->lx;
  
  insist (m);
  insist (n);
  insist (n->xbar);
  insist (mac_address);
  insist (map_address);
  insist (oiport >= 0);
  insist (oiport < LX_HOST_SIZE);
  
  if (lx_connected (n, port))
    return 1;
      
  mi_c (("got a reply from " lx_mac_format " port %d oiport %d",
         lx_mac_args (mac_address), port, oiport));
  
  if ((h = lx_find_host (lx, &m->hosts, mac_address, level, &higher)))
  {
    lx_node_t*o = mi_index2node (h->links [oiport].index);

    if (o)
    {
      mi_c (("we have seen "lx_mac_format " port %d before",
             lx_mac_args (mac_address), oiport));
      mi_c (("on " lx_node_format " port %d", lx_node_args (o),
             h->links [oiport].port));

      if (lx_map_merge (m, n, o, h->links [oiport].port - port) < 0)
      {
	mi_c (("merge failed. probably a duplicate board ID " lx_mac_format,
	       lx_mac_args (mac_address)));

	mi_syslog ("duplicate board ID " lx_mac_format, lx_mac_args (mac_address));

	lx->die = lx->aborted = 1;
      }
      return 0;
    }
    else
    {
      mi_c (("we have seen " lx_mac_format " before (but not port %d)",
             lx_mac_args (mac_address), oiport));
      lx_connect (n, port, h, oiport);
    }
  }
  else
  {
    lx_copy_route (r.hops,
                   lx_xbar_c (n)->routes [iport].hops,
                   lx_xbar_c (n)->routes [iport].length);
    r.length = lx_xbar_c (n)->routes [iport].length + 1;  
    insist (r.length > 0);   
    r.hops [r.length - 1] = port;

    if (lx_higher_than_me (lx, mac_address, level))
    {
      mi_c ((lx_mac_format " (level %d) is higher than me",
             lx_mac_args (mac_address), level));
	    
      mi_memcpy (m->winner_address, mac_address, 6);
      m->winner_route.length = r.length;
      m->winner_iport = iport;
      lx_copy_route (m->winner_route.hops, r.hops, r.length);
#if 0
      mi_c (("winner route is %s",
             lx_print_route (m->winner_route.hops, m->winner_route.length)));
#endif
      lx->aborted = 1;
      return 1;
    }
    h = lx_map_add_host (m, iport, n, port, oiport, host_type,
                         level, mac_address, higher);
    if (h == 0)
      return 0;
  }
  return 1;
  except: return 0;
}
 

static int
lx_map_find_hosts (lx_map_t*m, int iport, lx_node_t*n)
{
  unsigned char mac_address [6];
  unsigned char map_address [6];
  unsigned map_version;
  int i, j,  rr, port, host_type, level, flags, context, hop, oiport;
  lx_route_t r;
  lx_t*lx = m->lx;
  int sends;
  
  insist (m);
  insist (n);
  insist (n->xbar);
  insist (lx_xbar_c (n)->routes [iport].length < LX_ROUTE_SIZE);
  insist (iport >= 0);
  insist (iport < MI_MY_SIZE);

  mi_c (("looking for hosts on " lx_node_format, lx_node_args (n)));

  lx_advance_time (lx);

  for (j = 0; j < (int) lx->tries && !lx->aborted; j++)
  {
    sends = 0;  
    
    for (i = 1 - LX_XBAR_SIZE; i < LX_XBAR_SIZE && !lx->aborted; i++)
    {
      if (!lx_in_bounds (n, i) || lx_connected (n, i))
	continue;
    
      insist (!n->original.port);
      hop = i - n->original.port;

      if (hop < 1 - LX_XBAR_SIZE || hop >= LX_XBAR_SIZE)
	continue;

      lx_copy_route (r.hops,
                     lx_xbar_c (n)->routes [iport].hops,
		     lx_xbar_c (n)->routes [iport].length);
      r.length = lx_xbar_c (n)->routes [iport].length + 1;  
      insist (r.length > 0);
     
      r.hops [r.length - 1] = hop;
      port = i;

      rr = lx_send_scout (lx, iport, &r, lx->timestamp, LX_SCOUTING,
                          &oiport, &port, &host_type, mac_address,
			  &map_version, map_address, &level, &flags, &context);
      insist (rr);
    
      if (rr < 0 && !lx_map_found_host (m, n, iport, oiport, port, host_type,
                                        mac_address, map_version, map_address,
					level, flags, context))
	return 0;
      
      if (lx_wait_scout_reply (lx, iport, MI_POLL_USECS, lx->timestamp,
                               &oiport, &port, &host_type, mac_address,
			       &map_version, map_address, &level,
			       &flags, &context, 1) &&
	  !lx_map_found_host (m, n, iport, oiport, port, host_type,
	                      mac_address, map_version, map_address,
			      level, flags, context))
	return 0;
	
      sends++;
    }
    
    if (!sends)
      break;
    insist (!lx->duration);
    insist (lx->timeout);
    lx_set_alarm (lx, lx->timeout);
    
    while (lx_wait_scout_reply (lx, iport, 0, lx->timestamp, &oiport,
                                &port, &host_type, mac_address, &map_version,
				map_address, &level, &flags, &context, 0))
    {
      if (!lx_map_found_host (m, n, iport, oiport, port, host_type,
                              mac_address, map_version, map_address,
			      level, flags, context))
      {
	lx_clear_alarm (lx);
	return 0;
      }
      
      if (lx->aborted)
      {
	lx_clear_alarm (lx);
	return 1;
      }
    }
  }
  return 1;
  except: return 0;
}


static int
lx_map_can_match (lx_node_t*copy, lx_node_t*original, int offset)
{  
  insist (copy && original);
  insist (offset > -LX_XBAR_SIZE && offset < LX_XBAR_SIZE);
  
  return offset && !lx_get_node (original, offset);

  except: return 0;
}

#define lx_xbar2offset(lx, xbar) ((lx_xbar_t*) xbar - lx->xbar_block)

static lx_node_t *
lx_offset2xbar (lx_map_t*m, int offset)
{
  lx_node_t*n;
  insist (m && m->lx);
  insist (offset >= 0 && offset < LX_MAX_XBARS);
  
  n = (lx_node_t*) (m->lx->xbar_block + offset);
  insist (lx_queued (&m->frontier, n));
  return n;
  except: return 0;
}

static int
lx_map_match (lx_map_t*m, int iport, lx_node_t*n)
{
  int i, j, t;
  lx_node_t*o;
  lx_extended_t e;
  lx_route_t*r = (lx_route_t*) &e;
  int port, context;
  static int last = 0;
  
  lx_t*lx = m->lx;

  insist (m && n && n->xbar);
  insist (!lx_has_hosts (n));
  insist (iport >= 0 && iport < MI_MY_SIZE);
  insist (!lx->duration);
  
  lx_advance_time (lx);

  mi_c (("trying to match "  lx_node_format " timestamp %u",
	 lx_node_args (n), lx->timestamp));

  for (o = m->frontier.tail; o && !lx->aborted; o = mi_index2node (o->prev))
  {
    lx_copy_route (r->hops,
                   lx_xbar_c (n)->routes [iport].hops,
		   lx_xbar_c (n)->routes [iport].length);

    for (j = 0; j < LX_ACTIVE_TRIES + 1 && !lx->aborted; j++)
    {  
      for (i = 1 - LX_XBAR_SIZE; i < LX_XBAR_SIZE && !lx->aborted; i++)
      {
	if (!j && last) i = last;
	
	if (i && lx_map_can_match (n, o, -i))
	{
	  r->length = lx_xbar_c (n)->routes [iport].length +
	    lx_xbar_c (o)->routes [iport].length + 1;
	  r->hops [lx_xbar_c (n)->routes [iport].length] = i;
	  lx_reverse_route (r->hops + lx_xbar_c (n)->routes [iport].length + 1, 
			    lx_xbar_c (o)->routes [iport].hops,
			    lx_xbar_c (o)->routes [iport].length);
	  port = i;
	  context = (int)lx_xbar2offset (lx, o);
	  
	  t = lx_send_xbar (lx, iport, r, lx->timestamp, &port, &context);
	  if (t < 0)
	  {
	    last = port;
	    return lx_map_merge (m, n, lx_offset2xbar (m, context), -port);
	  }
	  else if (t == 0)
	  {
	    return 0;
	  }

	  insist (lx->timeout);
	  if (lx_wait_xbar (lx, iport, MI_POLL_USECS,
	                    lx->timestamp, &port, &context, 1))
	  {
	    last = port;
	    return lx_map_merge (m, n, lx_offset2xbar (m, context), -port);
	  }
	}
	if (!j && last) break;
      }
    }  
  }
  
  if (lx_wait_xbar (lx, iport, lx->timeout, lx->timestamp, &port, &context, 0))
    return lx_map_merge (m, n, lx_offset2xbar (m, context), -port);
  
  mi_c ((lx_node_format " is original", lx_node_args (n)));
  except: return 0;  
}

static int lx_map_find_xbars (lx_map_t*m, int iport, lx_node_t*n, int ports [2 * LX_XBAR_SIZE])
{
  int i, j;
  lx_extended_t e;
  lx_route_t*r = (lx_route_t*) &e;
  int port, context, hop;
  lx_t*lx = m->lx;
  int sends = 0;
  
  insist (m && n && n->xbar);
  insist (!lx->staged_xbars.count);
  insist (ports);
  insist (iport >= 0 && iport < MI_MY_SIZE);

  mi_c (("looking for xbars on " lx_node_format, lx_node_args (n)));

  if (lx_xbar_c (n)->routes [iport].length >= LX_EFFECTIVE_ROUTE_SIZE)
  {
    mi_c (("reached max diameter"));
    return 1;
  }

  for (i = 0; i < 2 * LX_XBAR_SIZE; i++)
    ports [i] = 0;

  while (lx->pendings [LX_SMALL_SEND])
    lx_receive (lx, 0, 0, 0, 0); 

  lx_advance_time (lx);

  for (j = 0; j < (int) lx->tries && !lx->aborted; j++)
  { 
    sends = 0;

    for (i = 1 - LX_XBAR_SIZE; i < LX_XBAR_SIZE && !lx->aborted; i++)
    {
      if (!lx_in_bounds (n, i) || lx_connected (n, i) || ports [i - (1 - LX_XBAR_SIZE)])
	continue;

      insist (!n->original.port);
      hop = i - n->original.port;

      if (hop < 1 - LX_XBAR_SIZE || hop >= LX_XBAR_SIZE)
	continue;

      r->length =
	lx_there_and_back_again (r->hops,
				 lx_xbar_c (n)->routes [iport].hops,
				 hop,
				 lx_xbar_c (n)->routes [iport].length);
      insist (r->length);
      port = i;

      if (lx_send_xbar (lx, iport, r, lx->timestamp, &port, &context) < 0)
      {
	insist (port >= 1 - LX_XBAR_SIZE);
	insist (port < LX_XBAR_SIZE);
	ports [port - (1 - LX_XBAR_SIZE)] = 1;
      }

      if (lx_wait_xbar (lx, iport, MI_POLL_USECS, lx->timestamp, &port,
                        &context, 1))
      {
	insist (port >= 1 - LX_XBAR_SIZE);
	insist (port < LX_XBAR_SIZE);
	ports [port - (1 - LX_XBAR_SIZE)] = 1;
      }
      sends++;
    }
    
    if (!sends)
      break;
    insist (!lx->duration);
    insist (lx->timeout);
    lx_set_alarm (lx, lx->timeout);

    while (lx_wait_xbar (lx, iport, 0, lx->timestamp, &port, &context, 0))
    {
      insist (port >= 1 - LX_XBAR_SIZE && port < LX_XBAR_SIZE);
      ports [port - (1 - LX_XBAR_SIZE)] = 1;
    }
  }
  return 1;
  except: return 0;
}

static int lx_map_found_xbar32 (lx_map_t*m, int iport, lx_node_t*n, int port, int id, int mask, int absolute_port)
{
  lx_node_t*nn;
  lx_t*lx = m->lx;
  
  insist (m && n);

  mi_c (("found x32 on port %d (absolute %d, id %u)", port, absolute_port, id));
  
  if (lx_connected (n, port))
  {
    mi_c (("port already connected"));
    return -1;
  }

  if ((nn = lx_find_xbar (lx, &m->xbars, id)) || (nn = lx_find_xbar (lx, &m->frontier, id)))
  {
    mi_c (("xbar %x already seen", id));
    
    if (!lx_connected (nn, absolute_port - lx_xbar_c (nn)->absolute_port))
    {
      lx_connect (n, port, nn, absolute_port - lx_xbar_c (nn)->absolute_port);
    }
    return -1;
  }
  
  nn = lx_map_add_xbar (m, iport, n, port, id, mask, absolute_port);

  mi_c (("adding " lx_node_format " to frontier", lx_node_args (nn)));
  lx_put (&m->frontier, nn);
  insist (m->frontier.tail == nn);

  return 1;
  except: return 0;
}

static int lx_map_find_xbar32s (lx_map_t*m, int iport, lx_node_t*n)
{
  int i, j, k;
  lx_extended_t e;
  lx_route_t*r = (lx_route_t*) &e;
  int port, context, hop, a;
  lx_t*lx = m->lx;
  int sends = 0;
  int id, mask, absolute_port, quadrant_enabled;

  char ports [LX_XBAR_SIZE * 2];

  insist (m && n && n->xbar);
  insist (!lx->staged_xbars.count);
  insist (iport >= 0 && iport < MI_MY_SIZE);

  mi_c (("looking for xbar 32s on " lx_node_format, lx_node_args (n)));

  if (lx_xbar_c (n)->routes [iport].length >= LX_EFFECTIVE_ROUTE_SIZE)
  {
    mi_c (("reached max diameter"));
    return 1;
  }

  while (lx->pendings [LX_SMALL_SEND])
    lx_receive (lx, 0, 0, 0, 0); 

  lx_advance_time (lx);

  for (j = 0; j < (int) lx->tries && !lx->aborted; j++)
  { 
    sends = 0;

    for (k = 0; k < 2 * LX_XBAR_SIZE; k++)
      ports [k] = 0;

    for (i = 1 - LX_XBAR_SIZE; i < LX_XBAR_SIZE && !lx->aborted; i++)
    {
      if (!lx_in_bounds (n, i) || lx_connected (n, i))
	continue;
      
      insist (!n->original.port);
      hop = i - n->original.port;

      insist (hop >= 1 - LX_XBAR_SIZE && hop < LX_XBAR_SIZE);

      r->length = lx_there_and_back_again (r->hops, lx_xbar_c (n)->routes [iport].hops, hop, lx_xbar_c (n)->routes [iport].length);
      insist (r->length);
      port = i;

      if (lx_send_xbar32 (lx, iport, r, lx->timestamp, &port, &context, &id, &mask, &absolute_port, &quadrant_enabled) < 0 || 
	  lx_wait_xbar32 (lx, iport, MI_POLL_USECS, lx->timestamp, &port, &context, &id, &mask, &absolute_port, &quadrant_enabled, 1))
      {
	insist (port >= 1 - LX_XBAR_SIZE);
	insist (port < LX_XBAR_SIZE);
	insist (id && mask);

	if (!(a = lx_map_found_xbar32 (m, iport, n, port, id, mask, absolute_port)))
	  return 0;

	if (a > 0)
	{
	  ports [port - (1 - LX_XBAR_SIZE)]++;
	  insist (ports [port - (1 - LX_XBAR_SIZE)] <= lx->tries);
	}
	
      }
      sends++;
    }
    
    if (!sends)
      break;
    insist (!lx->duration);
    insist (lx->timeout);

    while (lx_wait_xbar32 (lx, iport, lx->timeout, lx->timestamp, &port, &context, &id, &mask, &absolute_port, &quadrant_enabled, 0))
    {
      insist (port >= 1 - LX_XBAR_SIZE && port < LX_XBAR_SIZE);
      insist (id && mask);

      if (!(a = lx_map_found_xbar32 (m, iport, n, port, id, mask, absolute_port)))
	return 0;

      if (a > 0)
      {
	ports [port - (1 - LX_XBAR_SIZE)]++;
	insist (ports [port - (1 - LX_XBAR_SIZE)] <= lx->tries);
      }
    }

    for (i = 1 - LX_XBAR_SIZE; i < LX_XBAR_SIZE; i++)
    {
      if (ports [i - (1 - LX_XBAR_SIZE)])
      {
	lx_node_t*nn;
	
	insist (lx_in_bounds (n, i) && lx_connected (n, i));
	
	nn = lx_get_node (n, i);
	insist (nn && nn->xbar && lx_xbar_c (nn)->id);
	
	a = lx_map_find_hosts (m, iport, nn);
      } 
    }
  }
  return 1;
  except: return 0;
}

static int
lx_map_verify_disconnected (lx_map_t*m, int iport)
{
  int i, did_match;
  lx_t*lx = m->lx;
  unsigned char mac_address [6];
  unsigned char map_address [6];
  int rr, port, oiport, context, host_type, level, flags;
  unsigned map_version;
  lx_route_t r;
  
  insist (m && lx);
  insist (m->map_valid);
  insist (iport >= 0 && iport < MI_MY_SIZE);
  insist (m->root && !lx_get_node (m->root, iport));
  
  lx_set_timeout (lx, LX_ACTIVE_TIMEOUT, lx->active_tries);
  r.length = 0;

  for (i = 0; i < (int) lx->tries && !lx->aborted; i++)
  {  
    rr = lx_send_scout (lx, iport, &r, lx_advance_time (lx),
                        LX_SCOUTING, &oiport, &port, &host_type,
			mac_address, &map_version, map_address,
			&level, &flags, &context);
    insist (rr);
    
    if (rr < 0 ||
        lx_wait_scout_reply (lx, iport, lx->timeout, lx->timestamp, &oiport,
	                     &port, &host_type, mac_address, &map_version,
			     map_address, &level, &flags, &context, 0))
    {
      mi_c (("found host " lx_mac_format, lx_mac_args (mac_address)));
      return 0;
    }
  }
  
  r.length = 1;
  r.hops [0] = 0;
  
  for (i = 0; i < (int) lx->tries && !lx->aborted; i++)
  {  
    port = 0;
    
    rr = lx_send_xbar (lx, iport, &r, lx_advance_time (lx), &port, &context);
    insist (rr);
    
    if (rr < 0 ||
        lx_wait_xbar (lx, iport, lx->timeout, lx->timestamp,
	              &port, &context, 0))
    {
      mi_c (("found xbar"));
      return 0;
    }
  }
  did_match = m->kids_match;
  m->kids_match = 1;  
  if (!did_match)
    lx_map_update_state (lx, 1, m);

  return 1;
  except: return 0;
}


static int lx_map_explore (lx_map_t*m, int iport)
{
  int ports [2 * LX_XBAR_SIZE];
  unsigned char mac_address [6];
  unsigned char map_address [6];
  lx_node_t*n, *nn;
  lx_route_t r;
  int j, i, rr, port, oiport, context, host_type, level, flags;
  unsigned map_version;
  lx_t*lx = m->lx;
  
  insist (m);
  insist (!m->root);
  insist (iport >= 0 && iport < MI_MY_SIZE);
  
  r.length = 0;

  m->root = lx_new_host (m, iport, iport, lx->mac_address,
                         lx->host_type, lx->level, &r);
  insist (m->root);
 
  lx_put (&m->hosts, m->root);

  mi_c (("looking for directly connected host"));  

  port = 0;

  lx_advance_time (lx);
  
  for (j = 0; j < (int) lx->tries && !lx->aborted; j++)
  {  
    rr = lx_send_scout (lx, iport, &r, lx->timestamp, LX_SCOUTING, &oiport,
                        &port, &host_type, mac_address, &map_version,
			map_address, &level, &flags, &context);
    insist (rr);
    
    if (rr < 0 ||
        lx_wait_scout_reply (lx, iport, lx->timeout, lx->timestamp, &oiport,
	                     &port, &host_type, mac_address, &map_version,
			     map_address, &level, &flags, &context, 0))
    {
      mi_c (("found host " lx_mac_format, lx_mac_args (mac_address)));
    
      if (lx_memcmp (lx_host_c (m->root)->mac_address, mac_address, 6))
      {
	if (lx_higher_than_me (lx, mac_address, level))
	{
	  mi_c ((lx_mac_format " (level %d) is higher than me",
	         lx_mac_args (mac_address), level));
	    
	  mi_memcpy (m->winner_address, mac_address, 6);
	  m->winner_route.length = r.length;
	  m->winner_iport = iport;
	  m->winner_route.length = 0;
	  lx->aborted = 1;
	  return 0;
	}
	else
	{
	  n = lx_new_host (m, iport, oiport, mac_address, host_type, level, &r);
	  lx_put (&m->hosts, n);
	
	  return lx_connect (m->root, iport, n, oiport);
	}
      }
      else return lx_connect (m->root, iport, m->root, iport);
    }
    if (lx->aborted)
      return 0;
  }
  
  mi_c (("looking for first xbar"));

  r.length = 1;
  r.hops [0] = 0;
  lx_host_c (m->root)->routes [iport][iport].length = r.length;
  lx_copy_route (lx_host_c (m->root)->routes [iport] [iport].hops,
                 r.hops,
		 r.length);

  lx_advance_time (lx);

  n = 0;

  mi_c (("looking for directly connected xbar32"));

  for (j = 0; j < (int) lx->tries && !lx->aborted; j++)
  {  
    int absolute_port, quadrant_enabled, mask;
    int id;
    port = 0;

    rr = lx_send_xbar32 (lx, iport, &r, lx->timestamp, &port, &context, &absolute_port, &quadrant_enabled, &mask, &id);
    insist (rr);

    if (rr < 0 || lx_wait_xbar32 (lx, iport, lx->timeout, lx->timestamp, &port, &context, &id, &mask, &absolute_port, &quadrant_enabled, 0))
    {

      mi_c (("found x32  (absolute %d, id %u)", absolute_port, id));
      insist (absolute_port >= 0 && absolute_port < LX_XBAR_SIZE);
      r.length = 0;
      lx_connect (m->root, iport, (n = lx_new_xbar (m, iport, &r, id, mask, absolute_port)), port);
      break;
    }
  }

  if (!n)
  {
    mi_c (("no x32 found, looking for an x16 instead"));

    lx_advance_time (lx);
    
    for (j = 0; j < (int) lx->tries && !lx->aborted; j++)
    {  
      port = 0;

      rr = lx_send_xbar (lx, iport, &r, lx->timestamp, &port, &context);
      insist (rr);
      
      if (rr < 0 || lx_wait_xbar (lx, iport, lx->timeout, lx->timestamp, &port, &context, 0))
      {
	mi_c (("found xbar"));
	r.length = 0;
	lx_connect (m->root, iport, (n = lx_new_xbar (m, iport, &r, 0, 0, 0)), port);
	break;
      }
    }
  }
  
  if (n)
  {
    /*there was an xbar. explore.*/

    insist (!lx_get (&m->frontier));
    rr = lx_map_find_hosts (m, iport, n);
    insist (rr);
    
    lx_put (&m->frontier, n);

    while (!lx->aborted && (n = m->frontier.head))
    {
      if (!lx_map_find_xbar32s (m, iport, n) || !lx_map_find_xbars (m, iport, n, ports))
	return 0;

      for (i = 1 - LX_XBAR_SIZE; i < LX_XBAR_SIZE; i++)
      {
	if (lx_connected (n, i) || !ports [i - (1 - LX_XBAR_SIZE)])
	  continue;

	if (!(nn = lx_map_add_xbar (m, iport, n, i, 0, 0, 0)))
	  return 0;

	if (lx_map_find_hosts (m, iport, nn))
	{
	  if (lx_has_hosts (nn) || !lx_map_match (m, iport, nn))
	  {
	    mi_c (("adding " lx_node_format " to frontier",
		   lx_node_args (nn)));
	    lx_put (&m->frontier, nn);
	    insist (m->frontier.tail == nn);
	  }
	}
	mi_punt (lx);

	if (m->frontier.head != n)
	{
	  mi_c (("xbar I was exploring was deleted"));
	  break;
	}
      }
      if (m->frontier.head == n)
      {
	/*try and find some missing hosts*/
	lx_map_find_hosts (m, iport, n);

	if (m->frontier.head == n && !lx_put (&m->xbars, lx_get (&m->frontier)))
	  return 0;
      }
    }
      
    insist (!m->merge_queue.head);
    return !lx->aborted;
  }

  return -1;
  except: return 0;
}

static int
lx_map_query_new_version (lx_t*lx, int iport, unsigned*map_version,
			  unsigned char map_address [6])
{
  int i, r, port, oiport, host_type, level, flags, context;
  unsigned char mac_address [6];
  unsigned char parent_address [6];
  lx_map_t*map;

  insist (lx && map_version && map_address);
  insist (iport >= 0 && iport < MI_MY_SIZE);
  
  map = lx_get_map (lx, iport);
  insist (map);

  mi_memcpy (parent_address, map->parent_address, 6);
  
  lx_advance_time (lx);
  lx_set_timeout (lx, LX_VERIFY_CONNECTED_TIMEOUT, LX_VERIFY_CONNECTED_TRIES);

  for (i = 0; i < lx->tries; i++)
  { 
    r = lx_send_scout (lx, iport, &map->winner_route, lx->timestamp,
		       LX_QUERYING_MAP_VERSION, &oiport, &port, &host_type,
		       mac_address, map_version, map_address, &level, &flags,
		       &context);
    insist (r);
    if (r < 0 ||
	lx_wait_scout_reply (lx, iport, lx->timeout, lx->timestamp, &oiport,
			     &port, &host_type, mac_address, map_version,
			     map_address, &level, &flags, &context, 0))
    {      
      if (*map_version &&
	  !lx_zero (map_address, 6) &&
	  (*map_version != map->map_version ||
	   lx_memcmp (map_address, map->map_address, 6)))
      {
	mi_c ((lx_mac_format " has a new map (" lx_mac_format " version %d)",
	       lx_mac_args (map->winner_address),
	       lx_mac_args (map_address), 
	       *map_version));
	
	if (flags & LX_FLAG_MAP_VALID)
	  return 1;
	
	mi_c (("but it is invalid."));
	return -1;
      }
      if (!lx_higher_than_me (lx, mac_address, level))
      {
	mi_c (("winner changed to a loser"));
	return 0;
      }
      return -1;
    }
    if (lx_memcmp (map->parent_address, parent_address, 6))
    {
      mi_c (("my parent has changed"));
      return -1;
    }
  }
  mi_c (("no reply from " lx_mac_format,  lx_mac_args (map->winner_address)));
  except: return 0;
}

static int
lx_map_check_for_xbar (lx_t*lx, int iport, lx_node_t*n, int p,
                       int hop, int connected, int fast)
{
  int port;
  int j, rr, context;
  lx_map_t*map = lx_get_map (lx, iport);
  lx_extended_t e;
  lx_route_t*r = (lx_route_t*) &e;
  lx_node_t*o;
  int id, mask, absolute_port, quadrant_enabled;
  
  insist (n);
  insist (lx);
  insist (n->xbar);
  insist (iport >= 0);
  insist (iport < MI_MY_SIZE);
  insist (map);

  if (!map->map_valid)
    return 0;

  insist (p > -LX_XBAR_SIZE);
  insist (p < LX_XBAR_SIZE);
  insist (hop > -LX_XBAR_SIZE);
  insist (hop < LX_XBAR_SIZE);  
  
  o = lx_get_node (n, p);
  insist (o || !connected);
  
  if (!fast && connected)
    lx_set_timeout (lx, LX_VERIFY_CONNECTED_TIMEOUT,
                    LX_VERIFY_CONNECTED_TRIES);
  else
    lx_set_timeout (lx, LX_VERIFY_DISCONNECTED_TIMEOUT,
                    LX_VERIFY_DISCONNECTED_TRIES);

  lx_advance_time (lx);

  r->length = lx_there_and_back_again (r->hops,
                                       lx_xbar_c (n)->routes [iport].hops,
				       hop,
				       lx_xbar_c (n)->routes [iport].length);
  insist (r->length);

  for (j = 0; j < (int) lx->tries; j++)
  {
    port = p;

    /*if it's an x32, send an x32 message*/
    if (o && lx_xbar_c (o)->id)
    {
      rr = lx_send_xbar32 (lx, iport, r, lx->timestamp, &port, &context, &id, &mask, &absolute_port, &quadrant_enabled);
      insist (rr);
	
      if (rr < 0 || lx_wait_xbar32 (lx, iport, lx->timeout, lx->timestamp, &port, &context, &id, &mask, &absolute_port, &quadrant_enabled, 0))
      {
	/*got a message. return 1 if it's what we expect*/
	if (id != lx_xbar_c (o)->id)
	{
	  mi_c (("id changed from %x to %x", lx_xbar_c (o)->id, id));
	  return 0;
	}
	return 1;
      }
    }
    else if (o)
    {
      /*it is an x16.*/
      rr = lx_send_xbar (lx, iport, r, lx->timestamp, &port, &context);
      insist (rr);
      
      if (rr < 0 || (lx_wait_xbar (lx, iport, lx->timeout, lx->timestamp, &port, &context, 0) && port == p))
	return 1;
    }
    else
    {
      /*disconnected port. look for an x32, then an x16*/
      rr = lx_send_xbar32 (lx, iport, r, lx->timestamp, &port, &context, &id, &mask, &absolute_port, &quadrant_enabled);
      insist (rr);
	
      if ((rr < 0 || lx_wait_xbar32 (lx, iport, lx->timeout, lx->timestamp, &port, &context, &id, &mask, &absolute_port, &quadrant_enabled, 0)) && port == p)
      {
	mi_c (("found x32 on port %d (absolute %d, id %u)", port, absolute_port, id));
	return 1;
      }
      rr = lx_send_xbar (lx, iport, r, lx->timestamp, &port, &context);
      insist (rr);
      
      if (rr < 0 || (lx_wait_xbar (lx, iport, lx->timeout, lx->timestamp, &port, &context, 0) && port == p))
	return 1;
    }
  }
  except: return 0;
}

static int
lx_map_check_xbar_new_hosts (lx_t*lx, int iport, lx_node_t*n)
{
  int i, j, context, hop;
  lx_extended_t e;
  int port, oiport, host_type, level, flags, last_port;
  unsigned char mac_address [6], map_address [6];
  unsigned map_version;
  lx_map_t*map = lx_get_map (lx, iport);  
  lx_route_t*r = (lx_route_t*) &e;
  lx_node_t*t;
  
  insist (n && n->xbar);
  insist (lx);
  insist (iport >= 0 && iport < MI_MY_SIZE);
  insist (map);
  
  if (!map->map_valid)
    return 0;

  mi_c (("checking for new hosts on " lx_node_format, lx_node_args (n)));

  lx_set_timeout (lx, LX_VERIFY_DISCONNECTED_TIMEOUT,
                  LX_VERIFY_DISCONNECTED_TRIES);
  lx_advance_time (lx);

  t = lx_follow (map->root, iport, &lx_xbar_c (n)->routes [iport], &last_port);
  insist (t == n);
  insist (last_port > -LX_XBAR_SIZE && last_port < LX_XBAR_SIZE);
  
  for (i = 0; i < (int) lx->tries; i++)
  {
    for (j = 1 - LX_XBAR_SIZE; j < LX_XBAR_SIZE; j++)
    {  
      hop = j - last_port;

      if (hop > -LX_XBAR_SIZE && hop < LX_XBAR_SIZE && !lx_get_node (n, j))
      {   
	r->length = 0;
	lx_append_route (r, &lx_xbar_c (n)->routes [iport], hop);
	insist (r->length);
	port = j;
	
	if (lx_send_scout (lx, iport, r, lx->timestamp, LX_VERIFYING, &oiport,
	                   &port, &host_type, mac_address, &map_version,
			   map_address, &level, &flags, &context) < 0)
	{
	  mi_c (("new host " lx_mac_format " on port %d",
	         lx_mac_args (mac_address), port));
	  return 0;
	}
      }
    }
  }
  if (lx_wait_scout_reply (lx, iport, lx->timeout, lx->timestamp, &oiport,
                           &port, &host_type, mac_address, &map_version,
			   map_address, &level, &flags, &context, 0))
  {
    mi_c (("new host " lx_mac_format " on port %d",
           lx_mac_args (mac_address), port));
    return 0;
  }
  return 1;
  except: return 0;
}


static int
lx_map_check_xbars_up (lx_t*lx, int iport, int max_level)
{
  int p, hop, there, last_port;
  lx_node_t*n,*before;
  lx_map_t *m = lx_get_map (lx, iport);
  
  insist (lx);
  insist (iport >= 0 && iport < MI_MY_SIZE);
  insist (m);

  if (!m->map_valid)
    return 0;

  insist (m->num_xbars);
  
  before = lx_get_node (m->root, iport);
  
  for (;;)
  {
    insist (before && before->xbar);
    n = lx_follow (m->root, iport,
                   &lx_xbar_c (before)->routes [iport], &last_port);
    insist (n == before);
    insist (last_port > -LX_XBAR_SIZE && last_port < LX_XBAR_SIZE);
    insist (before && before->xbar);

    do
    {
      p = lx_xbar_c (before)->last++ % (LX_XBAR_SIZE * 2 - 2) +
	1 - LX_XBAR_SIZE;
      hop = p - last_port;
    } while (hop <= -LX_XBAR_SIZE || hop >= LX_XBAR_SIZE);

    if ((n = lx_get_node (before, p)) && n->xbar)
    {
      if ((m->xbar_verify_flag == LX_FLAG_XBAR_VERIFY_CLOS &&
           lx_xbar_c (before)->level > lx_xbar_c (n)->level) ||
	  (!m->xbar_verify_flag == LX_FLAG_XBAR_VERIFY_UP_DOWN &&
	   lx_xbar_c (before)->level < lx_xbar_c (n)->level))
	return 1;
    }
    
    if (!n || n->xbar)
    {
      mi_c (("checking %s link %d on " lx_node_format,
             n ? "connected" : "disconnected", p, lx_node_args (before)));

      there = lx_map_check_for_xbar (lx, iport, before, p, hop, n ? 1 : 0, 0);

      if (!m->map_valid)
	return 0;

      if (!n && there)
      {
	mi_c (("new link"));
	lx_set_change (lx, iport, 0, 0, LX_NEW_LINK, m->map_version,
	               m->map_address, lx->mac_address, 0, before->index, p + before->first, lx_xbar_c (before)->id);
	return 0;
      }
      if (n && !there)
      {
	mi_c (("missing link"));
	lx_set_change (lx, iport, 0, 0, LX_MISSING_LINK, m->map_version,
	               m->map_address, lx->mac_address, 0, before->index, p + before->first, lx_xbar_c (before)->id);
	return 0;
      }
    }
    if (!n || !n->xbar)
      return 1;
    if (m->xbar_verify_flag == LX_FLAG_XBAR_VERIFY_CLOS &&
        lx_xbar_c (n)->level == max_level)
      return 1;
    
    before = n;
  }
  except: return 0;
}

static int
lx_map_check_xbars_all (lx_t*lx, int iport)
{
  int i, p, hop, there, num_ports, ports_per_host, first, last, last_port;
  lx_node_t*n, *o;
  lx_map_t *m = lx_get_map (lx, iport);
  
  insist (lx);
  insist (iport >= 0 && iport < MI_MY_SIZE);
  insist (m);

  if (!m->map_valid)
    return 0;

  insist (m->num_xbars && m->num_hosts);
  insist (m->map_valid);
  insist (m->root);
  
  num_ports = m->num_xbars * (LX_XBAR_SIZE * 2 - 1);
  ports_per_host = 1 + num_ports / m->num_hosts;
  first = ports_per_host * m->root->index;
  if ((last = first + ports_per_host) >= num_ports)
    last = num_ports - 1;
  
  mi_c (("ports per host is %d, checking global xbar ports %d to %d",
         ports_per_host, first, last));
  
  for (i = first; i <= last; i++)
  {
    insist (i / (LX_XBAR_SIZE * 2) >= 0);
    insist (i / (LX_XBAR_SIZE * 2) < m->num_xbars);
    n = mi_index2node (m->xbar_array [i / (LX_XBAR_SIZE * 2)]);
    insist (n && n->xbar);
    
    mi_punt (lx);
    
    o = lx_follow (m->root, iport, &lx_xbar_c (n)->routes [iport], &last_port);
    insist (n == o);
    insist (last_port > -LX_XBAR_SIZE && last_port < LX_XBAR_SIZE);

    p = i % (LX_XBAR_SIZE * 2 - 2) + 1 - LX_XBAR_SIZE;
    insist (p >= 1 - LX_XBAR_SIZE);
    insist (p < LX_XBAR_SIZE);
    hop = p - last_port;
    if (hop && hop > -LX_XBAR_SIZE && hop < LX_XBAR_SIZE)
    {
      if (!(o = lx_get_node (n, p)) || o->xbar)
      {	
	mi_c (("checking %s link %d on " lx_node_format,
	       o ? "connected" : "disconnected", p, lx_node_args (n)));

	there = lx_map_check_for_xbar (lx, iport, n, p, hop, n ? 1 : 0, 1);

	if (!m->map_valid)
	  return 0;

	if (!o && there)
	{
	  mi_c (("new link"));
	  lx_set_change (lx, iport, 0, 0, LX_NEW_LINK,
	                 m->map_version, m->map_address, lx->mac_address,
			 0, n->index,  p + n->first, lx_xbar_c (n)->id);
	  return 0;
	}
	if (o && !there)
	{
	  mi_c (("missing link"));
	  lx_set_change (lx, iport, 0, 0, LX_MISSING_LINK, m->map_version,
	                 m->map_address, lx->mac_address, 0, n->index, p + n->first, lx_xbar_c (n)->id);
	  return 0;
	}
      }
    }
  }
  return 1;
  except: return 0;
}

static int
lx_map_check_xbars (lx_t*lx, int iport, int max_level)
{
  lx_map_t *m = lx_get_map (lx, iport);  
  insist (lx);
  insist (iport >= 0 && iport < MI_MY_SIZE);
  insist (m);

  if (!m->map_valid)
    return 0;

  if (!m->num_xbars)
    return 1;

  switch (m->xbar_verify_flag)
  {
    case LX_FLAG_XBAR_VERIFY_CLOS:
    case LX_FLAG_XBAR_VERIFY_UP_DOWN:
      return lx_map_check_xbars_up (lx, iport, max_level);
      break;
    case LX_FLAG_XBAR_VERIFY_NONE:
      return 1;
      break;
    case LX_FLAG_XBAR_VERIFY_ALL:
      return lx_map_check_xbars_all (lx, iport);
      break;
    default:
      insist (0);
  }
  except: return 0;
  
}

static int
lx_map_check_xbars_new_hosts (lx_t*lx, int iport)
{
  int i;
  lx_map_t *m = lx_get_map (lx, iport);
  
  insist (lx);
  insist (iport >= 0 && iport < MI_MY_SIZE);
  insist (m);

  if (!m->map_valid)
    return 0;

  insist (m->map_valid);
  insist (m->root);
  
  for (i = m->root->index; i < m->num_xbars; i += m->num_hosts)
  {
    lx_node_t*n = mi_index2node (m->xbar_array [i]);
    insist (n && n->xbar && n->index == i);
    if (!lx_map_check_xbar_new_hosts (lx, iport, n))
      return 0;
  }
  
  return 1;
  except: return 0;
}


static int
lx_map_check_host (lx_t*lx, int iport, int ooiport, lx_node_t*n,
                   int reason, int*kids_match, int*matches_me, int*context)
{
  int r, j;
  unsigned char mac_address [6];
  unsigned char map_address [6];
  int port, oiport, host_type, level, flags, last_port;
  unsigned map_version;
  lx_map_t*map = lx_get_map (lx, iport);
  lx_node_t*t;
  
  insist (iport >= 0 && iport < MI_MY_SIZE);
  insist (ooiport >= 0 && ooiport < LX_HOST_SIZE);
  insist (map && context);
  
  if (!map->map_valid)
    return 0;
  
  if (lx_host_c (n)->host_type == LX_DEBUG_TYPE /*|| n == lx->map.root*/)
    return *kids_match = *matches_me = 1;
  
  if (!lx_get_node (n, ooiport))
    return -1;
  
  insist (lx && n && !n->xbar);
  insist (reason == LX_VERIFYING || reason == LX_TREE);

  insist (kids_match && matches_me);

  /*mi_c (("root is " lx_node_format, lx_node_args (lx->map.root)));*/

  lx_advance_time (lx);

  t = lx_follow (map->root, iport,
                 &lx_host_c (n)->routes [iport][ooiport], &last_port);
  
  insist (t == n);
  insist (last_port == ooiport);

  mi_c (("checking host " lx_node_format " his port %d",
         lx_node_args (n), ooiport));
  
  for (j = 0; j < (int) lx->tries && map->map_valid; j++)
  {
    port = 0;
    r = lx_send_scout (lx, iport, &lx_host_c (n)->routes [iport][ooiport],
                       lx->timestamp, reason, &oiport, &port, &host_type,
		       mac_address, &map_version, map_address,
		       &level, &flags, context);
    insist (r);    
    
    if (r < 0 ||
        lx_wait_scout_reply (lx, iport, lx->timeout, lx->timestamp, &oiport,
	                     &port, &host_type, mac_address, &map_version,
			     map_address, &level, &flags, context, 0))
    {
      if (!map->map_valid)
	return 0;
      
      if (host_type != lx_host_c (n)->host_type ||
          level != lx_host_c (n)->level ||
	  lx_memcmp (mac_address, lx_host_c (n)->mac_address, 6) ||
	  oiport != ooiport)
      {
	mi_c (("host changed " lx_mac_format, lx_mac_args (mac_address)));
	mi_c ((lx_mac_format " map_address " lx_mac_format
	       " map_version %d flags %d oiport %d" ,
	       lx_mac_args (mac_address), lx_mac_args (map_address),
	       map_version, flags, oiport));

	lx_set_change (lx, iport, 0, 0, LX_CHANGED_HOST, map->map_version,
	               map->map_address, lx->mac_address,
		       lx_host_c (n)->mac_address, n->index, ooiport, 0);
	return 0;
      }
      *matches_me = (map_version == map->map_version &&
                     lx_memcmp (map_address, map->map_address, 6) == 0);
      if (!*matches_me)
      {
	mi_c (("does not match me. map_version %d, map_address " lx_mac_format,
	       map_version, lx_mac_args (map_address)));
      }
      
      if (!(flags & LX_FLAG_MAP_VALID) &&
	  map_version == map->map_version &&
	  lx_memcmp (map_address, map->map_address, 6) == 0)
      {
	mi_c (("host map invalid"));
	return 0;
      }

      *kids_match = ((flags & LX_FLAG_KIDS_MATCH) != 0);

      mi_c (("kids %s", *kids_match ? "match" : "do not match"));
 
      return 1;
    }
  }

  if (map->map_valid)
  {
    mi_c (("missing host " lx_node_format, lx_node_args (n)));
    lx_set_change (lx, iport, 0, 0, LX_MISSING_HOST, map->map_version,
                   map->map_address, lx->mac_address,
		   lx_host_c (n)->mac_address, n->index, ooiport, 0);
  }
  
  except: return 0;
}


static int
lx_map_check_hosts (lx_t*lx, int iport, int initial)
{
  int i, j, r;
  int kids_match, did_match;
  int matches_me;
  int num_kids = 0;
  int num_matches = 0;
  int context;
  
  lx_map_t*map = lx_get_map (lx, iport);

  insist (lx);
  insist (iport >= 0);
  insist (iport < MI_MY_SIZE);
  insist (map);
  
  if (!map->map_valid)
    return 0;
  
  insist (map->num_hosts);
  insist (map->root);

  mi_c ((lx_node_format " checking hosts", lx_node_args (map->root)));

  lx_set_timeout (lx, LX_VERIFY_CONNECTED_TIMEOUT, LX_VERIFY_CONNECTED_TRIES);

  if (map->root->index)
  {
    lx_node_t*kid;

    kid = mi_index2node (map->host_array [(map->root->index - 1) / LX_BRANCH]);
    insist (kid);
    
    for (j = 0; j < LX_HOST_SIZE; j++)
    {
      if (!lx_map_check_host (lx, iport, j, kid, LX_VERIFYING, &kids_match,
                              &matches_me, &context))
	return 0;
    }
  }

  for (i = 0; i < LX_BRANCH; i++)
  {
    int n;

    if ((n = map->root->index * LX_BRANCH + 1 + i) < map->num_hosts)
    {
      lx_node_t*kid = mi_index2node (map->host_array [n]);
      insist (kid);

      num_kids++;
      insist (lx->thrash ||
              lx_hostcmp (lx->mac_address, lx->level,
	                  lx_host_c (kid)->mac_address,
			  lx_host_c (kid)->level) > 0);

      for (j = 0; j < LX_HOST_SIZE; j++)
      {
	r = lx_map_check_host (lx, iport, j, kid, LX_TREE,
	                       &kids_match, &matches_me, &context);
	if (r == 0)
	  return 0;
	
	if (r < 0) 
	  continue;
      }
      if (kids_match && matches_me)
	num_matches++;
    }
  }
  did_match = map->kids_match;
  map->kids_match = num_matches == num_kids;
  
  if (did_match != map->kids_match)
    lx_map_update_state (lx, iport, map);
  
  return 1;
  except: return 0;
}

static int
lx_map_collect_map (lx_t*lx, lx_map_t*map, int iport, lx_route_t*map_route,
                    unsigned char map_address [6], int map_version)
{
  int first, num_nodes, i, r = 0;
  int num_hosts, num_xbars;
  lx_map_reply_message_t*rm;
  int x32;
  
  rm = (lx_map_reply_message_t*)&lx->map_reply_buffer;

  insist (lx);
  insist (map_address);
  insist (iport >= 0);
  insist (iport < MI_MY_SIZE);
  insist (map_route);
  insist (!map->map_valid);
  insist (map);
  
  num_nodes = 0;
  
  lx_set_timeout (lx, LX_COLLECT_TIMEOUT, LX_COLLECT_TRIES);

  num_xbars = LX_MAX_XBARS;
  num_hosts = LX_MAX_HOSTS;
  
  mi_c (("collecting map on iport %d from " lx_mac_format,  iport,
         lx_mac_args (map->winner_address)));

  for (first = 0; first < num_hosts;)
  { 
    lx_advance_time (lx);

    for (i = 0; i < lx->tries; i++)
    {
      if (lx_send_map_request (lx, iport, map_route, lx->timestamp,
                               LX_MAP_HOST, first))
      {
	r = lx_wait_map_reply (lx, iport, lx->timeout, lx->timestamp,
	                       LX_MAP_HOST, rm);
	if (r != 0)
	  break;
      }
    }
    
    if (i == lx->tries)
    {
      mi_c (("no reply from " lx_mac_format,
             lx_mac_args (map->winner_address)));
      return 0;
    }    

    insist (r >= (int) sizeof (lx_old_map_reply_message_t));
    
    mi_c (("got map reply message %d", mi_htonl (rm->header.timestamp)));

    if (lx_memcmp (rm->map_address, map_address, 6) ||
        rm->map_version != (unsigned) mi_htonl (map_version))
    {
      mi_c (("map address / map version changed to " lx_mac_format " %u",
             lx_mac_args (rm->map_address), mi_htonl (rm->map_version)));
      return 0;
    }
    
    if (first == 0)
    {
      num_hosts = mi_htons (rm->num_hosts);
      num_xbars = mi_htons (rm->num_xbars);
      
      insist (num_hosts && num_hosts < LX_MAX_HOSTS);
      insist (rm->num_nodes);
      
      insist (map->num_xbars == 0 && map->num_hosts == 0);
      lx_before_receiving (map, num_hosts, num_xbars);
      map->routing_flag = rm->routing_flag;
      map->xbar_verify_flag = rm->xbar_verify_flag;
      map->seek_size = mi_htonl (rm->header.context);
    }
    else
    {
      insist (num_hosts == mi_htons (rm->num_hosts));
      insist (num_xbars = mi_htons (rm->num_xbars));
    }
    
    num_nodes = mi_htons (rm->num_nodes);
    
    for (i = 0; i < num_nodes; i++, first++)
    {     
      lx_node_t*n = mi_index2node (map->host_array [first]);
      insist (n);
      insist (first < num_hosts);

      mi_c (("reading " lx_node_format, lx_node_args (n)));
      lx_net2host (map, &rm->nodes.host [i], lx_host_c (n),
                   num_hosts, num_xbars);
      
      insist (!map->hosts.tail ||
              lx_hostcmp (lx_host_c (map->hosts.tail)->mac_address,
	                  lx_host_c (map->hosts.tail)->level,
			  lx_host_c (n)->mac_address,
			  lx_host_c (n)->level) > 0);
      lx_put (&map->hosts, lx_remove_from_anywhere (n));
      mi_punt (map->lx);
    }
  }

  if (!(map->root = lx_bsearch_host (map, lx->mac_address, lx->level)))
  { 
    mi_c (("my %s has changed", map->root ? "level" : "existence"));
    mi_c (("Therefore this map version is invalid"));
    mi_memcpy (map->map_address, map_address, 6);
    map->map_version = map_version;
    map->map_valid = 0;
    lx_map_update_state (lx, iport, map);
    lx_set_change (lx, iport, 0, 0, LX_CHANGED_ORDER_IN_COLLECT,
                   map_version, map_address, lx->mac_address, 0, 0, 0, 0);
    return 0;
  }
  
  for (first = 0; first < num_xbars;)
  {
    lx_advance_time (lx);

    for (i = 0; i < lx->tries; i++)
    {
      if (lx_send_map_request (lx, iport, map_route,
                               lx->timestamp, LX_MAP_XBAR, first))
      {
	r = lx_wait_map_reply (lx, iport, lx->timeout,
			       lx->timestamp, LX_MAP_XBAR, rm);
	if (r != 0)
	  break;
      }
    }
    
    if (i == lx->tries)
    {
      mi_c (("no reply from " lx_mac_format,
             lx_mac_args (map->winner_address)));
      return 0;
    }    

    x32 = rm->type == LX_MAP_XBAR32;
    
    insist (r >= (int) sizeof (lx_old_map_reply_message_t));

    mi_c (("got map reply %d message", mi_htonl (rm->header.timestamp)));
    
    if (lx_memcmp (rm->map_address, map_address, 6) || rm->map_version != (unsigned) mi_htonl (map_version))
    {
      mi_c (("map address / map version changed to " lx_mac_format " %u",
             lx_mac_args (rm->map_address), mi_htonl (rm->map_version)));
      return 0;
    }

    insist (num_hosts == mi_htons (rm->num_hosts));
    insist (num_xbars = mi_htons (rm->num_xbars));
    
    num_nodes = mi_htons (rm->num_nodes);
    insist (num_nodes);
    
    for (i = 0; i < num_nodes; i++, first++)
    {
      lx_node_t*n = mi_index2node (map->xbar_array [first]);
      insist (n);
      insist (first < num_xbars);

      mi_c (("reading " lx_node_format, lx_node_args (n)));
      if (x32)
	lx_net2xbar (map, &rm->nodes.xbar [i], lx_xbar_c (n), num_hosts, num_xbars);
      else
	 lx_net2xbar16 (map, &rm->nodes.xbar16 [i], lx_xbar_c (n), num_hosts, num_xbars);

      lx_put (&map->xbars, lx_remove_from_anywhere (n));
      mi_punt (map->lx);
    }
  }

  if (!lx_get_node (map->root, iport))
  {
    mi_c (("this map doesn't have my port %d", iport));
    mi_c (("Therefore this map version is invalid"));
    mi_memcpy (map->map_address, map_address, 6);
    map->map_version = map_version;
    map->map_valid = 0;
    lx_map_update_state (lx, iport, map);
    lx_set_change (lx, iport, 0, 0, LX_CHANGED_PORT_IN_COLLECT,
                   map_version, map_address, lx->mac_address,
		   map->winner_address, 0, 0, 0);
    return 0;
  }
  
  return 1;
  except: return 0;
}

static int lx_map_route (lx_t*lx, int iport)
{
  int i, j, num_spines;
  lx_map_t*m = 0;

  insist (lx);  
  insist (iport >= 0 && iport < MI_MY_SIZE);
  insist (lx->num_routes);

  for (i = 0; i < lx->num_ports; i++)
  {
    m = lx_get_map (lx, i);
    mi_punt (m->lx);

    if (!m->routeable)
      continue;
    lx_routing_forget (m);

  }
  
  for (i = 0; i < lx->num_ports; i++)
  {
    mi_punt (m->lx);
      
    m = lx_get_map (lx, i);
    insist (m);
    if (!m->routeable)
      continue;
    
    insist (m->root);

    lx_number_xbars_bfs (m, i);
    
    if (m->routing_flag == LX_FLAG_ROUTING_CAREFUL)
    {
      m->max_level = lx_routing_number_xbars_clos (m, &num_spines);
    }

    for (j = 0; j < lx->num_routes; j++)
    {  
      int r;

      mi_punt (m->lx);

      /* routing not yet done */
      r = 0;

      /* Try careful routes, may fail if clos */
      if (m->routing_flag == LX_FLAG_ROUTING_CAREFUL &&
	  m->max_level != 0)
      {
	r = lx_routing_careful_routes (m, i, m->root, m->seek_size,
				       LX_ROUTING_LEAST, 1);

      }

      /* If not routed yet, try shortest path */
      if (r == 0)
      {
	/* If we ever ran careful routing, need to renumber */
	if (m->routing_flag == LX_FLAG_ROUTING_CAREFUL)
	{
	  lx_number_xbars_bfs (m, -1);
	}

	m->max_level = 0;

	/* If shortest path fails, give up */
	if (!lx_routing_shortest_path (m, i, m->root,
	                               m->routing_flag ==
				       LX_FLAG_ROUTING_UP_DOWN))
	  return 0;
      }
      
      /* If route_filebase is set, override computed routes with
       * ideal routes from a file, but only if they work.
       */
      mi_check_file_routes(m, m->root, i, j);
      
      if (!lx_set_routes (m, i, j))
      {
	return 0;
      }
    }
  }
  return 1;
  except: return 0;
}

static int
lx_map_port (lx_t*lx, int iport)
{
  int r;
  lx_map_t*m = lx_get_map (lx, iport);

  insist (lx);
  insist (iport >= 0 && iport < MI_MY_SIZE);
  insist (m->active);
  insist (!m->map_valid);
  
  lx->aborted = 0;
  mi_c (("mapping port %d", iport));
  
  while (!lx->aborted && !lx->die)
  {
    m->map_valid = 0;
    lx_reset_map (lx, m);
    lx_set_timeout (lx, LX_ACTIVE_TIMEOUT, lx->active_tries);
    lx_map_update_state (lx, iport, m);

    if ((r = lx_map_explore (m, iport)))
    {
      if (r < 0)
      {
	mi_c (("mapper is disconnected (on port %d)", iport));
	insist (m->root && !lx_get_node (m->root, iport));
	mi_c (("it doesn't matter?"));
      }
      if (lx->extra_hosts)
	lx_inflate_map (m, iport, 8, lx->extra_hosts);
      r1= 0;
      lx_before_sending (m);
      if (r1)race++;
      
      if (!++m->map_counter)
	m->map_counter = 1;
      m->map_version = m->map_counter;
      
      mi_c (("map version is now %d", m->map_version));
      mi_memcpy (m->map_address, lx->mac_address, 6);
      m->map_valid = 1;
      lx_map_check_other (lx, m);
      lx_map_update_state (lx, iport, m);
      return 1;
    }
  }
  mi_c (("going passive"));

  m->active = 0;
  m->map_valid = 0;
  lx_reset_map (lx, m);
  lx_bzero ((char*) m->parent_address, 6);
  lx_map_update_state (lx, iport, m);

  return 1;  
  except: return 0;
}

static int
lx_map_report_change (lx_t*lx, int iport, unsigned char parent_address [6],
                      lx_route_t*parent_route)
{
  int j, r;
  int oiport, port, host_type, level, flags, context;
  unsigned char mac_address [6];
  unsigned char map_address [6];
  unsigned map_version;

  insist (lx);
  insist (iport >= 0);
  insist (iport < MI_MY_SIZE);
  insist (parent_address);
  insist (parent_route);
  insist (lx->changes [iport].pending >= 0);
  
  if (!lx->changes [iport].pending)
    return 1;

  lx_set_timeout (lx, LX_ACTIVE_TIMEOUT, lx->active_tries);
  lx_advance_time (lx);

  for (j = 0; j < (int) lx->tries; j++)
  {
    port = 0;
    r = lx_send_change_report (lx, iport, &lx->changes [iport],
                               parent_address, parent_route, lx->timestamp);
    insist (r);
    
    if (r < 0 ||
        lx_wait_scout_reply (lx, iport, lx->timeout, lx->timestamp,
	                     &oiport, &port, &host_type, mac_address,
			     &map_version, map_address, &level,
			     &flags, &context, 0))
    {
      lx->changes [iport].pending = 0;
      return 1;
    }
  }
  lx->changes [iport].pending--;
  
  return 0;
  except: return 0;
}

static int
lx_map_poll (lx_t*lx, int iport)
{
  int r;
  lx_map_t*m = lx_get_map (lx, iport);
  unsigned map_version;
  unsigned char map_address [6];
  
  insist (lx);
  insist (iport >= 0 && iport < MI_MY_SIZE);
  
  if (!lx_zero (m->parent_address, 6))
  { 
    mi_memcpy (m->winner_address, m->parent_address, 6);
    m->winner_route.length = m->parent_route.length;
    m->winner_iport = m->parent_iport;
    lx_copy_route (m->winner_route.hops,
                   m->parent_route.hops, m->parent_route.length);
  }
  
  if (lx_zero (m->winner_address, 6))
    return !lx->level;

  if (m->winner_iport != iport)
  {
    if (!lx_map_shared (lx))
    {
      mi_c (("port flopped"));
      return !lx->level;
    }
    return 1;
  }

  lx_map_report_change (lx, iport, m->winner_address, &m->winner_route);

  if (!(r = lx_map_query_new_version (lx, iport, &map_version, map_address)))
  {
    mi_c (("query new_version failed"));
    return 0;
  }

  if (lx_memcmp (m->winner_address, m->parent_address, 6))
  {
    /*mi_c (("not listening. I'm not listening."));*/
    return 1;
  }

  if (r > 0)
  {
    lx_bzero ((char*) m->map_address, 6);
    m->map_version = 0;
    m->map_valid = 0;
    lx_map_update_state (lx, iport, m);

    lx_reset_map (lx, m);
    if (!lx_map_collect_map (lx, m, iport, &m->winner_route,
                             map_address, map_version))
    {
      mi_c (("collect map failed. I'll try again."));
      lx_reclaim_staged (lx);
      return 1;
    }    

    lx_before_sending (m);
    mi_memcpy (m->map_address, map_address, 6);
    m->map_version = map_version;
    m->map_valid = 1;
    m->active = 0;
    lx_map_check_other (lx, m);
    lx_map_update_state (lx, iport, m);
  } 
  return 1;
  except: return 0;
}

static int lx_map_verify (lx_t*lx, int iport)
{
  lx_map_t*m = lx_get_map (lx, iport);
  
  insist (lx);
  insist (iport >= 0 && iport < MI_MY_SIZE);
  insist (m->num_hosts && m->root);
  insist (m->map_valid);
  insist (m->root);

  mi_c ((lx_mac_format " verifying port %d (%s)", lx_mac_args (lx->mac_address),
         iport, m->active ? "active" : "passive"));
  mi_c (("map_version %d, map_address " lx_mac_format, m->map_version,
         lx_mac_args (m->map_address)));
  mi_c (("%d host%s, %d xbar%s", m->num_hosts, m->num_hosts == 1 ? "" : "s",
         m->num_xbars, m->num_xbars == 1 ? "" : "s"));
  mi_c (("mapping %s done everywhere on port %d",
         m->done_everywhere ? "is" : "is not", iport));
  
  if ((!lx_get_node (m->root, iport) &&
       !lx_map_verify_disconnected (m, iport)) ||
      (lx_get_node (m->root, iport) &&
       (!lx_map_check_hosts (lx, iport, 0) ||
        !lx_map_check_xbars (lx, iport, m->max_level) ||
	!lx_map_check_xbars_new_hosts (lx, iport))) ||
      !m->map_valid)
  {
    mi_c (("verify failed"));
    return 0;
  }

  if (m->active)
  {
    int done_everywhere = m->kids_match;
    if (done_everywhere != m->done_everywhere)
    {
      m->done_everywhere = done_everywhere;
      lx_map_update_state (lx, iport, m);
    }
  }
  return m->map_valid;
  except: return 0;
}

static void
lx_map_invalidate (lx_t*lx, int iport)
{
  int i, shared;
  lx_map_t*m, *o;
  
  insist (lx);
  insist (iport >= 0 && iport < MI_MY_SIZE);

  shared = lx_map_shared (lx);

  for (i = 0; i < MI_MY_SIZE; i++)
  {
    if (i != iport) continue;
    
    m = &lx->maps [i];
    
    if (!lx->level)
    {
      lx_bzero ((char*) m->winner_address, 6);
      lx_bzero ((char*) m->parent_address, 6);
    }
    m->map_valid = 0;
    m->done_everywhere = 0;

    if (shared)
    {
      o = lx_map_get_other (lx, m);
      insist (o);
      o->map_valid = 0;
      mi_memcpy (o->map_address, m->map_address, 6);
      o->map_version = m->map_version;
    }
    lx_map_update_state (lx, iport, m);
  }
  except:;
}


void
lx_mapper_main (lx_t*lx)
{ 
  int i, j, r, thrash, t, num_spines, collected;
  unsigned map_version;
  unsigned char map_address [6];
  lx_map_t*m;

  thrash = 0;

  lx_init (lx);

  mi_get_address (lx, lx->mac_address);
  insist (!lx_zero (lx->mac_address, 6));
  
  mi_c (("I am host " lx_mac_format " (type %d)",
         lx_mac_args (lx->mac_address), lx->host_type));

  mi_c (("%d hosts, %d xbars per map message piece",
         lx_piece_count (lx_map_host_t), lx_piece_count (lx_map_xbar_t)));
  mi_c (("longest route supported is %d hops", LX_EFFECTIVE_ROUTE_SIZE));
  mi_c (("%d hosts and %d xbars max", LX_MAX_HOSTS, LX_MAX_XBARS));
  
  lx_advance_time (lx);
  
  for (i = 0; i < lx->num_ports; i++)
  {
    m = &lx->maps [i];
    m->map_version = m->map_counter = mi_rand ();
    m->active = lx->level;
    mi_c (("going %s on port %d", m->active ? "active" : "passive", i));

    /* set an initial mapper state */
    lx_map_update_state (lx, i, m);
  }

  for (;;)
  {    
    for (i = 0; i < lx->num_ports; i++)
    { 
      m = lx_get_map (lx, i);
      
      map_version = m->map_version;
      mi_memcpy (map_address, m->map_address, 6);

      if (m->map_valid)
      {
	if (!lx_map_verify (lx, i))
	  lx_map_invalidate (lx, i);
      }
      else if (m->active)
      {
	insist (m == &lx->maps [i]);
	m->mapping = 1;
	t = mi_time (lx);
	if (lx_map_port (lx, i))
	{
	  t = (mi_time (lx) - t) / (1000 * 1000);
	  mi_c (("mapping took %d sec%s", t, t == 1 ? "" : "s"));
	}
	m->mapping = 0;
      }
      collected = 0;
      
      if (!m->active)
      {
	if (!lx_map_poll (lx, i))
	{
	  m = &lx->maps [i];
	  m->map_valid = 0;
	  lx_bzero ((char*) m->map_address, 6);

	  m->active = lx->level;

	  if (!lx->level)
	  {
	    lx_bzero ((char*) m->winner_address, 6);
	    lx_bzero ((char*) m->parent_address, 6);
	  }
	  else
	  {  
	    mi_c (("going %s on port %d", m->active ? "active" : "passive", i));
	  }
	  lx_map_update_state (lx, i, m);
	}
	else if (lx_get_map (lx, i) != m)
	{
	  m = lx_get_map (lx, i);
	  collected = 1;
	}
      }
      
      if (m->map_valid)
      {
	for (j = 0; j < lx->num_ports; j++)
	{
	  lx_map_t*mm = lx_get_map (lx, j);
	  insist (mm);
	  mm->routeable = mm->map_valid;
	}
    
	if (m->map_version != map_version ||
	    lx_memcmp (m->map_address, map_address, 6) || collected)
	{
	  mi_c (("%d host%s and %d xbar%s", m->num_hosts,
	         m->num_hosts == 1 ? "" : "s",
		 m->num_xbars, m->num_xbars == 1 ? "" : "s"));
	  m->kids_match = 0;
	  m->done_everywhere = 0;
	  mi_dump_map (lx, i, m);
	  lx_map_update_state (lx, i, m);

	  thrash = 0;

	  if (m->xbar_verify_flag == LX_FLAG_XBAR_VERIFY_CLOS)
	    m->max_level = lx_routing_number_xbars_clos (m, &num_spines);
	  else if (m->xbar_verify_flag == LX_FLAG_XBAR_VERIFY_UP_DOWN)
	    lx_number_xbars_bfs (m, i);
	  
	  if (!lx_map_route_nodes (lx, m))
	  {
	    d_rf++;
	    lx_map_invalidate (lx, i);
	    lx_set_change (lx, i, 0, 0, LX_FAILED_ROUTING, m->map_version,
	                   m->map_address, lx->mac_address,
			   lx->mac_address, 0, 0, 0);
	    continue;
	  }

	  if (!lx_map_check_hosts (lx, i, 1))
	  {
	    mi_c (("initial host check failed"));
	    lx_map_invalidate (lx, i);
	    continue;
	  }

	  mi_set_route_begin (lx);
	  m->routing = 1;
	  
	  t = mi_time (lx);
	  if (!(r = lx_map_route (lx, i)))
	  {
	    d_rf++;
	    lx_map_invalidate (lx, i);
	    lx_set_change (lx, i, 0, 0, LX_FAILED_ROUTING, m->map_version,
	                   m->map_address, lx->mac_address,
			   lx->mac_address, 0, 0, 0);
	  }
	  
	  t = (mi_time (lx) - t) / (1000 * 1000);
	  mi_c (("routing took %d sec%s", t, t == 1 ? "" : "s"));
	  d_rt = t;
	  
	  m->routing = 0;
	  t = mi_time (lx);
	  mi_set_route_end (lx);

	  t = (mi_time (lx) - t) / (1000 * 1000);
	  mi_c (("route loading took %d sec%s", t, t == 1 ? "" : "s"));
	  d_et = t;

	  if (m->map_valid)
	  {
	    if (m->xbar_verify_flag == LX_FLAG_XBAR_VERIFY_CLOS)
	      m->max_level = lx_routing_number_xbars_clos (m, &num_spines);
	    else if (m->xbar_verify_flag == LX_FLAG_XBAR_VERIFY_UP_DOWN)
	      lx_number_xbars_bfs (m, i);

	    r = lx_map_route_nodes (lx, m);
	    insist (r);
	  }
	}
        if (lx->thrash && ++thrash == lx->thrash)
	{
	  thrash = 0;
	  m->map_valid = 0;
	  if (!(lx->level = mi_rand () & 0xff))
	    lx->level = 1;
	  lx_map_update_state (lx, i, m);
	}
      }
      if (lx->die)
	return;
      if (lx->map_once && lx_map_check_done_everywhere (lx))
      {
	mi_c (("map-once enabled. exiting"));
	return;
      }
      
    }
    lx->aborted = 0;

    do
      lx_wait (lx, LX_PASSIVE_TIMEOUT, (int*)&lx->aborted, 0);
    while (lx->paused || lx->net_paused);

  }
  insist (0);
  except:;
}
